package jcircus.translator;

import java.io.File;
import java.io.FileWriter;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import jcircus.environment.ChanInfoEnv;
import jcircus.environment.ChanUseEnv;
import jcircus.environment.Environment;
import jcircus.environment.ProcChanEnv;
import jcircus.environment.ProcChanUseEnv;
import jcircus.environment.TypeList;
import jcircus.exceptions.FailParsingException;
import jcircus.exceptions.FailTranslationException;
import jcircus.exceptions.FailTypeCheckingException;
import jcircus.exceptions.InvalidParameterException;
import jcircus.exceptions.InvalidSubTypeException;
import jcircus.exceptions.NotYetImplementedException;
import jcircus.exceptions.JCircusException;
import jcircus.exceptions.UnrecoveredErrorException;
import jcircus.util.CircusType;
import jcircus.util.CodeFormatting;
import jcircus.util.Constants;
import jcircus.util.ChanUse;
import jcircus.util.MathToolkitConstants;
import jcircus.util.ProcInfo;
import jcircus.util.Util;
import jcircus.visitor.EnvLoadingVisitor;
import jcircus.visitor.TranslatorVisitor;
import net.sourceforge.czt.base.ast.TermA;
import net.sourceforge.czt.circus.ast.CircusFactory;
import net.sourceforge.czt.circus.ast.CircusProcess;
import net.sourceforge.czt.circus.ast.ParamProcess;
import net.sourceforge.czt.circus.ast.ProcessPara;
import net.sourceforge.czt.z.ast.AxPara;
import net.sourceforge.czt.z.ast.Branch;
import net.sourceforge.czt.z.ast.Decl;
import net.sourceforge.czt.z.ast.DeclName;
import net.sourceforge.czt.z.ast.Expr;
import net.sourceforge.czt.z.ast.Freetype;
import net.sourceforge.czt.z.ast.Para;
import net.sourceforge.czt.z.ast.RefExpr;
import net.sourceforge.czt.z.ast.Spec;
import net.sourceforge.czt.z.ast.VarDecl;
import net.sourceforge.czt.z.ast.ZSect;
import net.sourceforge.czt.z.util.Factory;
import net.sourceforge.czt.z.util.ZString;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.exception.MethodInvocationException;
import org.apache.velocity.exception.ParseErrorException;
import org.apache.velocity.exception.ResourceNotFoundException;
import circparser.formw.process.IteratedProcess;
import circparser.formw.util.ColonDecl;
import circparser.formw.util.Declaration;

/**
 * Translator2Java.java
 *
 * @author  Angela Freitas
 */
public class Translator2Java {
    
    private String projectDir;
    private String projectName;
    private Spec spec;
    private TranslatorVisitor translatorVisitor_;
    private EnvLoadingVisitor envLoadingVisitor_;
    private Environment environment_;
    private List<ProcInfo> procInfoList;
    private List axParaCodeList;
    
    /**
     * Constructor.
     *
     */
    public Translator2Java(String projectDir, String projectName, Spec spec) {
        this.projectDir = projectDir;
        this.projectName = projectName;
        this.spec = spec;
        this.environment_ = new Environment();
        this.envLoadingVisitor_ = new EnvLoadingVisitor(this.environment_);
        this.translatorVisitor_ = new TranslatorVisitor(this.environment_);
        this.procInfoList = new ArrayList();
        this.axParaCodeList = new ArrayList();
    }
    
    public List<ProcInfo> getProcInfoList() {
        return this.procInfoList;
    }
    
    public Environment getEnvironment() {
        return this.environment_;
    }
    
    /**
     * Main method of this class.
     *
     * @param projectDir
     * @param projectName
     * @param spec
     * @param nameProjectRun
     * @throws Exception
     */
    public void translate(boolean gui) throws Exception {
        
        projectDir = projectDir + "\\" + projectName + "\\" + Constants.DIR_SOURCE;
        
        try {
            // Loads the environment and annotates names with NameType information
            this.envLoadingVisitor_.visitSpec(spec);
            
        } catch (UnrecoveredErrorException e) {
            // There was a fatal error
            throw new FailTranslationException(envLoadingVisitor_.getErrors(), e);
        }
        
        if (!envLoadingVisitor_.getErrors().isEmpty()) {
            // There was not a fatal error, but errors need to be reported
            throw new FailTranslationException(envLoadingVisitor_.getErrors());
        }

        Iterator iterator = ((ZSect) spec.getSect().get(0)).getPara().iterator();
        Para para;
        boolean createMain;
        
        // Iterates over the paragraphs to translate and load information about 
        // process paragraphs and axiomatic definitions
        while (iterator.hasNext()) {

            para = (Para) iterator.next();

            if (para instanceof ProcessPara) {
                
                DeclName procName = ((ProcessPara) para).getDeclName();
                CircusProcess circusProcess = ((ProcessPara) para).getCircusProcess();
                
                String procParaCode = null;
                try {
                    // Translates the process definition
                    procParaCode = (String) para.accept(translatorVisitor_);
                    
                } catch (UnrecoveredErrorException e) {
                    
                    // There was a fatal error
                    throw new FailTranslationException(translatorVisitor_.getErrors(), e);
                }
                
                if (!translatorVisitor_.getErrors().isEmpty()) {
                    // There was not a fatal error
                    throw new FailTranslationException(translatorVisitor_.getErrors());
                }
                
                // If this is not being run with GUI, a class Main will be created
                // only for those processes which are not parameterized
                if (!gui && !(circusProcess instanceof ParamProcess)) {
                    createMain = true;
                } else {
                    createMain = false;
                }
                
                // Stores information about the process definition
                ProcInfo procInfo = new ProcInfo(procName.toString(), (ProcessPara) para,
                        procParaCode, getNormalizedParams(circusProcess), createMain);
                
                procInfoList.add(procInfo);

            } else if (para instanceof AxPara) {
                
                // Translates the axiomatic definition
                String axParaCode = (String) para.accept(translatorVisitor_);
                axParaCodeList.add(axParaCode);
            }
        }
        
        // Generate source code directly if this is not being run with GUI
        if (!gui) {
            createSources();
        }                    
    }
    
    /**
     * This method is the implementation of
     * 
     * |[ CircusProgram ]|^{Program} proj
     * 
     * in the strategy, where:
     *
     * - CircusProgram is the program defined by the attribute Spec
     * - proj is the name of the project, defined by the attribute "projectName"
     */
    public void createSources() throws Exception {
        
        // Create the directories for the project
        this.createDirectoriesStructure();
        
        // Corresponds to function "DeclareTypeClass" in the strategy
        this.createTypeClass();
        
        // Creates class that represents the basic type \arithmos (number)
        this.createCircusIntegerClass();
        
        // Corresponds to function |[ ]|^{FreeTypes} in the strategy 
        this.createClassesForFreeTypes();
        
        // Corresponds to function "DeclareAxDefClass" in the strategy
        this.createAxDefHorizDefClass();

        // Implements "|[ ProcDecls ]|^{ProcDecls} proj" in the strategy
        for (int i = 0; i < procInfoList.size(); i++) {
            ProcInfo procInfo = procInfoList.get(i);
            
            // Class for process
            createProcessClass(procInfo.getProcessName(), procInfo.getCode());

            // Main for process
            if (procInfo.getCreateMain()) {
                createClassMain(procInfo);
                createGUI(procInfo);
                createBatFile(procInfo);
            }
            
        }        
        System.out.println("Source files for project \'" + projectName + "\' were created at " + projectDir);
    }
    
    /**
     * Gets the parameters of a process in a list of VarDecl.
     */
    private List<VarDecl> getNormalizedParams(CircusProcess circusProcess) {
        
        Factory factory = new Factory();
        List<VarDecl> normParams = new ArrayList();
        
        if (circusProcess instanceof ParamProcess) {
            
            List decls = ((ParamProcess) circusProcess).getDecl();
            for (int i = 0; i < decls.size(); i++) {
                Decl decl = (Decl) decls.get(i);
                if (decl instanceof VarDecl) {
                    List declNames = ((VarDecl) decl).getDeclName();
                    Expr expr = ((VarDecl) decl).getExpr();
                    for (int j = 0; j < declNames.size(); j ++) {
                        DeclName declName = (DeclName) declNames.get(j);
                        VarDecl newDecl = factory.createVarDecl(factory.list(declName), expr);
                        
                        normParams.add(newDecl);
                    }
                } else 
                    throw new InvalidParameterException("");
            }
        }
        return normParams;
    }
    
    /**
     * Creates the project folders.
     *
     * @param projectDir
     * @param projectName
     * @throws Exception
     */
    private void createDirectoriesStructure() throws Exception {
        
        File dirProcesses = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_PROCESSES);
        File dirAxDefs = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_AXDEFS);
        File dirTyping = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_TYPING);
        File dirGui = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_GUI);
        
        try {
            dirProcesses.mkdirs();
            dirAxDefs.mkdirs();
            dirTyping.mkdirs();
            dirGui.mkdirs();
            
        } catch (SecurityException se) {
            throw new Exception("The creation of the directory structure was not possible.");
        }
    }
        
    /**
     * Creates class Main for a process definition.
     *
     * @param projectDir
     * @param projectName
     * @param processDefinitionMain
     * @throws Exception
     */
    private void createClassMain(ProcInfo procInfo) throws Exception {
        
        File file;
        FileWriter fileWriter;
        String fileName;
        VelocityContext velocityContext;
        Template template = null;
        
        String nameMain = "Main_" + procInfo.getProcessName();
        String declarations = declarationsCode(procInfo.getProcessPara());
        String processCall = this.processCallCode(procInfo);

        // Create file
        fileName = projectDir + "\\" + projectName + "\\" + nameMain + ".java";
        file = new File(fileName);
        fileWriter = new FileWriter(fileName);
        velocityContext = new VelocityContext();

        Velocity.init();
        velocityContext.put("package", this.getPackage(projectName, null));
        velocityContext.put("imports", this.getImports(projectName));
        velocityContext.put("proj", projectName);
        velocityContext.put("nameMain", nameMain);
        velocityContext.put("declarations", declarations);
        velocityContext.put("processCall", processCall);

        template = Velocity.getTemplate(Constants.TMP_MAIN);

        // Writes the file
        template.merge(velocityContext, fileWriter);
        fileWriter.close();
            
    }
    
    /**
     * Create class GUI for a process definition.
     */
    private void createGUI(ProcInfo procInfo) throws Exception {
        
        File file;
        FileWriter fileWriter;
        String fileName;
        VelocityContext velocityContext;
        Template template = null;
        
        // Create file
        String nameMain = procInfo.getProcessName();
        String nameGui = this.getNameGui(nameMain);
        fileName = projectDir + "\\" + projectName + "\\" + Constants.PKG_GUI + "\\" + nameGui + ".java";
        file = new File(fileName);
        fileWriter = new FileWriter(fileName);
        velocityContext = new VelocityContext();

        Velocity.init();
        velocityContext.put("package", this.getPackage(projectName, Constants.PKG_GUI));
        velocityContext.put("imports", this.getImports(projectName));
        velocityContext.put("className", nameGui);
        velocityContext.put("chanDecl", this.chanDeclGuiCode(procInfo.getProcessPara()));
        velocityContext.put("chanAssign", this.chanAssignGuiCode(procInfo.getProcessPara()));
        velocityContext.put("chanParam", this.chanParamGuiCode(procInfo.getProcessPara()));
        velocityContext.put("chanInit", this.chanInitGuiCode(procInfo.getProcessPara()));
        velocityContext.put("chanActPerf", this.chanActPerfGuiCode(procInfo.getProcessPara()));
        
        template = Velocity.getTemplate(Constants.TMP_GUI);
        
        // Writes the file
        template.merge(velocityContext, fileWriter);
        fileWriter.close();

    }    

    /**
     * Creates the .bat file for a process.
     */
    private void createBatFile(ProcInfo procInfo) throws Exception {
        
        File file;
        FileWriter fileWriter;
        String fileName;
        VelocityContext velocityContext;
        Template template = null;
        
        // Create file
        String processName = procInfo.getProcessName();
        
        fileName = projectDir + "\\" + this.getNameBat(processName) + ".bat";
        file = new File(fileName);
        fileWriter = new FileWriter(fileName);
        velocityContext = new VelocityContext();

        Velocity.init();
        velocityContext.put("proj", projectName);
        velocityContext.put("main", this.getNameMain(processName));
        
        template = Velocity.getTemplate(Constants.TMP_BAT);
        
        // Writes the file
        template.merge(velocityContext, fileWriter);
        fileWriter.close();
    }
    
    /**
     * Code for declaration of GUI components of a channel.
     */
    private String chanDeclGuiCode(ProcessPara processParaMain) {
        String code = "";
        
        String processName = processParaMain.getDeclName().toString();
        CircusProcess circusProcess = processParaMain.getCircusProcess();
        ChanInfoEnv channelMSEnv = Util.getChannelMSEnvAnn(processParaMain);
        ProcChanEnv procChanEnv = Util.getProcChanEnvAnn(processParaMain);
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();

        Iterator it = channelMSEnv.iteratorKeys();
        
        // for each channel
        while (it.hasNext()) {
            
            String channelName = (String) it.next();
            ProcChanUseEnv syncEnv = channelMSEnv.get(channelName);
            
            if (!syncEnv.isSync() && !chanUseEnvHid.containsKey(channelName)) {
                
                // The channel does not take part in a parallelism, so it will
                // be in the graphical interface
                code = code + "\n\nprivate final GeneralChannel " + channelName + ";";
                code = code + "\nprivate javax.swing.JButton btn_" + channelName + ";";
                code = code + "\nprivate javax.swing.JLabel lbl_" + channelName + ";";
                code = code + "\nprivate javax.swing.JTextField txt_" + channelName + ";";
            }
        }

        return code;
    }
    
    /**
     * Code for parameters in the constructor in a GUI component.
     */
    private String chanParamGuiCode(ProcessPara processParaMain) {
        String code = "";
        
        String processName = processParaMain.getDeclName().toString();
        CircusProcess circusProcess = processParaMain.getCircusProcess();
        ChanInfoEnv channelMSEnv = Util.getChannelMSEnvAnn(processParaMain);

        ProcChanEnv procChanEnv = Util.getProcChanEnvAnn(processParaMain);
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();
        
        Iterator it = channelMSEnv.iteratorKeys();
        
        // for each channel
        while (it.hasNext()) {
            
            String channelName = (String) it.next();
            ProcChanUseEnv syncEnv = channelMSEnv.get(channelName);

            if (!syncEnv.isSync() && !chanUseEnvHid.containsKey(channelName)) {

                // The channel does not take part in a parallelism, so it will
                // be in the graphical interface
                // And also it cannot be hidden, otherwise the gui cannot see it
                code = code + ", GeneralChannel " + channelName;
            }
        }
        return code;
    }

    /**
     * Code for the body of a constructor in the GUI component.
     */
    private String chanAssignGuiCode(ProcessPara processParaMain) {
        String code = "";
        
        String processName = processParaMain.getDeclName().toString();
        CircusProcess circusProcess = processParaMain.getCircusProcess();
        ChanInfoEnv channelMSEnv = Util.getChannelMSEnvAnn(processParaMain);
        ProcChanEnv procChanEnv = Util.getProcChanEnvAnn(processParaMain);
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();

        Iterator it = channelMSEnv.iteratorKeys();
        
        // for each channel
        while (it.hasNext()) {
            
            String channelName = (String) it.next();
            ProcChanUseEnv syncEnv = channelMSEnv.get(channelName);
            
            if (!syncEnv.isSync()&& !chanUseEnvHid.containsKey(channelName)) {
                
                // The channel does not take part in a parallelism, so it will
                // be in the graphical interface
                code = code + "\nthis." + channelName + " = " + channelName + ";";
            }
        }

        return code;
    }
    
    /**
     *
     */
    private String chanInitGuiCode(ProcessPara processParaMain) throws Exception {
        String code = "";
        
        String processName = processParaMain.getDeclName().toString();
        CircusProcess circusProcess = processParaMain.getCircusProcess();
        ChanInfoEnv channelMSEnv = Util.getChannelMSEnvAnn(processParaMain);
        ProcChanEnv procChanEnv = Util.getProcChanEnvAnn(processParaMain);
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();
        
        Iterator it = channelMSEnv.iteratorKeys();
        
        // for each channel
        while (it.hasNext()) {
            
            String channelName = (String) it.next();
            ProcChanUseEnv syncEnv = channelMSEnv.get(channelName);
            
            if (!syncEnv.isSync() && !chanUseEnvHid.containsKey(channelName)) {
                
                ChanUse chanUse = syncEnv.getChanUseGuiNotSyncChannel();
                CircusType circusType = procChanEnv.getNameTypeEnv().getCircusType(channelName);
                
                /**
                 * Error when I try to use velocity:
                 * org.apache.velocity.exception.ParseErrorException: Encountered "java" at line 5, column 37.
                 * Was expecting one of:
                 * "," ...
                 * ")" ...
                 * at org.apache.velocity.Template.process(Template.java:141)
                 *
                 */
                if (circusType.isSyncType()) {
                    code = code + this.initGuiChanSync(channelName);
/*                  Hashtable ht = new Hashtable();
                    ht.put("chanName", channelName);
                    code = code + Util.getCodeFromTemplate(Constants.TMP_INITGUICHSYNC, ht);*/
                } else {
                    String commTypeName = circusType.getJavaCircusTypeName();
                    
/*                    if (commTypeName.equals(Constants.CLS_CIRCINT)) {
                        commTypeName = ZString.ARITHMOS;
                    }
*/
                    
                    code = code + this.initGuiChanComm(channelName, 
                            chanUse.toStringGUI(), commTypeName);
/*                  Hashtable ht = new Hashtable();
                    ht.put("chanName", channelName);
                    ht.put("chanType", chanUse.toStringGUI());
                    ht.put("commType", circusType.getJavaCircusTypeName());
                    code = code + Util.getCodeFromTemplate(Constants.TMP_INITGUICHCOMM, ht);*/
                }
            }
        }
        
        return code;
    }
    
    /**
     *
     */
    private String chanActPerfGuiCode(ProcessPara processParaMain) throws Exception {
        String code = "";

        String processName = processParaMain.getDeclName().toString();
        CircusProcess circusProcess = processParaMain.getCircusProcess();
        ChanInfoEnv channelMSEnv = Util.getChannelMSEnvAnn(processParaMain);
        ProcChanEnv procChanEnv = Util.getProcChanEnvAnn(processParaMain);
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();
        
        Iterator it = channelMSEnv.iteratorKeys();
        
        // for each channel
        while (it.hasNext()) {
            
            String channelName = (String) it.next();
            ProcChanUseEnv syncEnv = channelMSEnv.get(channelName);
            
            if (!syncEnv.isSync() && !chanUseEnvHid.containsKey(channelName)) {
                
                ChanUse chanUse = syncEnv.getChanUseGuiNotSyncChannel();
                
                // TODO: Change this code when dealing with sync channels
                CircusType circusType = procChanEnv.getNameTypeEnv().getCircusType(channelName);
                
                // The channel does not take part in a parallelism, so it will
                // be in the graphical interface
/*
 * Velocity does not work here.              
 *
                Hashtable ht = new Hashtable();
                ht.put("chanName", channelName);
                ht.put("chanCommType", circusType.getJavaCircusTypeName());

                if (chanUse.equals(JavaTypeChannel.Input))
                    code = code + Util.getCodeFromTemplate(Constants.DIR_TEMPLATES + "chanActPerfWrite.vm", ht);
                else 
                    code = code + Util.getCodeFromTemplate(Constants.DIR_TEMPLATES + "chanActPerfRead.vm", ht);
*/                
                if (chanUse.equals(ChanUse.Input))
                    code = code + this.chanActPerfWrite(channelName, circusType);
                    
                else 
                    code = code + this.chanActPerfRead(channelName, circusType);
                
            }
        }
        
        return code;
    }
    
    /**
     * Pg. 165.
     *
     * @param projectDir
     * @param projectName
     */
    private void createTypeClass() throws Exception {
        
        File file;
        FileWriter fileWriter;
        VelocityContext velocityContext;
        Template template = null;
        String declTypeConstants;
        
        file = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_TYPING + "\\" + Constants.CLS_TYPE + Constants.JAVA_EXT);
        fileWriter = new FileWriter(file);
        
        velocityContext = new VelocityContext();
        Velocity.init();
        velocityContext.put("declTypeConstants", this.declareTypeConstants());
        velocityContext.put("package", this.getPackage(projectName, Constants.PKG_TYPING));
        
        template = Velocity.getTemplate(Constants.TMP_TYPE);
        
        template.merge(velocityContext, fileWriter);
        fileWriter.close();
    }
    
    /**
     * Create CircusInteger.class
     *
     * @param projectDir
     * @param projectName
     */
    private void createCircusIntegerClass() throws Exception {
        
        File file;
        FileWriter fileWriter;
        VelocityContext velocityContext;
        Template template = null;
        String declTypeConstants;
        
        File dir = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_TYPING);
        boolean t;
        try {
            t = dir.mkdirs();
        } catch (SecurityException se) {
            throw new Exception("The creation of the directory was not possible.");
        }
        
        file = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_TYPING + "\\" + Constants.CLS_CIRCINT + Constants.JAVA_EXT);
        fileWriter = new FileWriter(file);
        
        velocityContext = new VelocityContext();
        velocityContext.put("package", this.getPackage(projectName, Constants.PKG_TYPING));
        
        Velocity.init();
        
        template = Velocity.getTemplate(Constants.TMP_CIRCINT);
        
        template.merge(velocityContext, fileWriter);
        fileWriter.close();
    }
    
    /**
     * Creates class for a process definition.
     * Pg. 137.
     *
     * @param projectDir
     * @param projectName
     */
    private void createProcessClass(String processName, String processBody) throws Exception {
        
        File file;
        FileWriter fileWriter;
        VelocityContext velocityContext;
        Template template = null;
        
        processBody = CodeFormatting.format(processBody);
        
        file = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_PROCESSES + "\\" + processName + ".java");
        fileWriter = new FileWriter(file);
        
        velocityContext = new VelocityContext();
        Velocity.init();
        velocityContext.put(Constants.PROJECT_NAME, projectName);
        velocityContext.put(Constants.$_CLASS_BODY, processBody);
        velocityContext.put(Constants.$_CLASS_NAME, processName);
        velocityContext.put("package", this.getPackage(projectName, Constants.PKG_PROCESSES));
        velocityContext.put("imports", this.getImports(projectName));
        
        template = Velocity.getTemplate(Constants.TMP_PROCDECL);
        
        template.merge(velocityContext, fileWriter);
        fileWriter.close();
    }
    
    /**
     * Creates classes for all the free types.
     *
     */
    private void createClassesForFreeTypes() throws Exception {
        
        Freetype freetype;
        
        //Free types
        for (int i=0; i<this.environment_.freeTypesEnvironmentSize(); i++) {
            freetype = this.environment_.freeTypesEnvironmentGet(i);
            this.createClassForFreeType(freetype);
        }
        
    }
    
    /**
     * Creates class for a free type.
     * Pg. 167.
     *
     * @param freetype
     */
    private void createClassForFreeType(Freetype freetype) throws Exception {
        
        File file;
        FileWriter fileWriter;
        VelocityContext velocityContext;
        Template template = null;
        
        String name = freetype.getDeclName().toString();
        String constants = "";
        String toStringCode = "\nString result = null;\nswitch(super.getValue()) {";
        
        String constructorCode = "";
        
        Branch branch;
        String nameFreeTypeElement;
        List branches = freetype.getBranch();
        int index = 0;
        
        // Constants
        for (int i = 0; i < branches.size(); i++) {
            branch = (Branch) branches.get(i);
            nameFreeTypeElement = branch.getDeclName().toString();
            constants = constants + "\npublic static final int " + nameFreeTypeElement + " = " + index + ";";
            
            toStringCode += "\ncase " + nameFreeTypeElement + ": result = \"" + nameFreeTypeElement + "\";\nbreak;";
            
            constructorCode += "\nif (st.equals(\"" + nameFreeTypeElement + "\"))\n\tthis.setValue(" + index + ");";
            
            index++;
        }

        toStringCode += "\n}";
        toStringCode += "\nreturn result;";
        
        constants = constants + "\npublic static final int MIN_VALUE = 0;";
        constants = constants + "\npublic static final int MAX_VALUE = " + (branches.size() - 1) + ";";
        
        // Create file
        file = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_TYPING + "\\" + name + ".java");
        fileWriter = new FileWriter(file);
        
        velocityContext = new VelocityContext();
        Velocity.init();
        velocityContext.put(Constants.PROJECT_NAME, projectName);
        velocityContext.put(Constants.$_CLASS_NAME, name);
        velocityContext.put("constants", constants);
        velocityContext.put("toString", toStringCode);
        velocityContext.put("constructor", constructorCode);
        velocityContext.put("package", this.getPackage(projectName, Constants.PKG_TYPING));
        
        template = Velocity.getTemplate(Constants.TMP_SUBTYPE);
        
        template.merge(velocityContext, fileWriter);
        fileWriter.close();
        
    }
    
    /**
     * Creates class for the axiomatic definitions and horizontal definitions.
     */
    private void createAxDefHorizDefClass() throws Exception {
        
        File file;
        FileWriter fileWriter;
        VelocityContext velocityContext;
        Template template = null;
        
        String body = "";
        
        for (int i = 0; i < axParaCodeList.size(); i++) {
            body = body + axParaCodeList.get(i) + "\n";
        }
        
        // Create file
        file = new File(projectDir + "\\" + projectName + "\\" + Constants.PKG_AXDEFS + "\\" + Constants.CLS_AXDEFS + Constants.JAVA_EXT);
        fileWriter = new FileWriter(file);
       
        velocityContext = new VelocityContext();
        Velocity.init();
        velocityContext.put("package", this.getPackage(projectName, Constants.PKG_AXDEFS));
        velocityContext.put("imports", this.getImports(projectName));
        velocityContext.put("body", body);
        
        template = Velocity.getTemplate(Constants.TMP_AXDEFS);
        
        template.merge(velocityContext, fileWriter);
        fileWriter.close();
        
    }   
    
    /**
     * Returns Java code for the declaration of constants for all the types.
     * Pg. 166.
     *
     * @return
     */
    private String declareTypeConstants() {
        
        String javaCode = "";

        // Insert all the free types
        int i;
        for (i=0; i<this.environment_.freeTypesEnvironmentSize(); i++) {
            Freetype ft = environment_.freeTypesEnvironmentGet(i);
            javaCode += "\npublic static final int " + 
                    ft.getDeclName() + " = " + i + ";";
        }
        
        // Insert the CircusInteger type
        javaCode += "\npublic static final int " + Constants.CLS_CIRCINT + " = " + i + ";";

        // min and max constants 
        javaCode += "\n\npublic static final int MIN_TYPE_ID = 0;";
        javaCode += "\npublic static final int MAX_TYPE_ID = " + i + ";";
          
        return javaCode;
    }
    
    /**
     * Code for declaration of channels in the main class.
     */
    private String declarationsCode(ProcessPara processParaMain) {
        
        String code = "";
        
        String processName = processParaMain.getDeclName().toString();
        CircusProcess circusProcess = processParaMain.getCircusProcess();
        ChanInfoEnv multiSyncEnv = Util.getChannelMSEnvAnn(processParaMain);
        ProcChanEnv procChanEnv = Util.getProcChanEnvAnn(processParaMain);
    
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();
        
        // Gets process id
        Integer procId = Util.getIdCircusProcessAnn(circusProcess).getId();
        
        Iterator it = multiSyncEnv.iteratorKeys();
        
        // For each channel
        while(it.hasNext()) {
            
            String channelName = (String) it.next();
            ProcChanUseEnv chanMSEnv = multiSyncEnv.get(channelName);

            if (!chanUseEnvHid.containsKey(channelName)) {
            
                // Code for the declaration of the channel - only channels that
                // are not hidden can appear in the gui
                code = code + this.channelCode(channelName, procId, chanMSEnv);
            }
        }
        
        return code;
    }

    /**
     *  Code for declaration of one channel in the main class. Example of code:
     *
     *  final Any2OneChannel[] from_a = Any2OneChannel.create(3); // number of processes that take part in the MS - MS only 
     *
     *  final Any2OneChannel to_a = new Any2OneChannel();
     *
     *  Hashtable ht = new Hashtable();
     *  ht.put("GC_A", new Integer(1));
     *  ht.put("GC_B", new Integer(0));
     *  ht.put("GC_C", new Integer(2));
     *  final ChanInfo chanInfo = new ChanInfo(ht);
     *
     *  final GeneralChannel a = new GeneralChannel(
     *          to_a, 
     *          from_a, // MS only
     *          chanInfo,
     *          "GC_P");
     *
     * OBS: Variables are no longer 'final', I do not remeber why they were like this
     * before.
     *
     */
    private String channelCode(String channelName, Integer procId, 
            ProcChanUseEnv chanMSEnv) {
        
        String code = "";
        
        String toChannel = "to_" + channelName;
        String fromChannel = "from_" + channelName;
        //String ht = "ht_" + channelName;
        String chanInfo = "chanInfo_" + channelName;
        
        /**
         * The variables in the 
         *
         */
        
        if (chanMSEnv.isMultiSync()) {
            // from_channel declaration, if this channel takes part in multisynchronization
            code = code + "Any2OneChannel[] " + fromChannel + " = Any2OneChannel.create(" + 
                    chanMSEnv.getCardinality() + ");\n";
        }
        
        // to_channel declaration
        code = code + "Any2OneChannel " + toChannel + " = new Any2OneChannel();\n";

        // Returns a list that assings each process name to its id (the position in the list)
        List ids = Util.resolveUndefinedChannels(chanMSEnv, true);

        // hashtable declaration
        code = code + Constants.CLS_CHINFO + " " + chanInfo + " = new " + Constants.CLS_CHINFO + "();\n";
        
        for (int i = 0; i < ids.size(); i++) {
            
            Integer subProcId = (Integer) ids.get(i);
            code = code + chanInfo + ".put(new Integer(" + subProcId + "), new Integer(" + i + "));\n";
        }
        
        // general channel
        if (chanMSEnv.isMultiSync()) {
            // multisynchronization
            code = code + "GeneralChannel " + channelName + " = new GeneralChannel(\n"
                    + toChannel + ",\n" 
                    + fromChannel + ",\n"
                    + chanInfo + ",\n"
                    + "new Integer(" + procId + "));\n";
        } else {
            // not multisynchronization
            code = code + "GeneralChannel " + channelName + " = new GeneralChannel(\n"
                    + toChannel + ",\n" 
                    + chanInfo + ",\n"
                    + "new Integer(" + procId + "));\n";
        }

        return code;
    }    
     
    /**
     * Code for call to the main process.
     *
     */
    private String processCallCode(ProcInfo procInfo) throws Exception {
        
        String code = "";
        
        String processName = procInfo.getProcessName();
        CircusProcess circusProcess = procInfo.getProcessPara().getCircusProcess();
        ChanInfoEnv chanInfoEnv = Util.getChannelMSEnvAnn(procInfo.getProcessPara());
        Integer procId = Util.getIdCircusProcessAnn(circusProcess).getId();
        ProcessPara processPara = procInfo.getProcessPara();
        ProcChanEnv procChanEnv = Util.getProcChanEnvAnn(processPara);
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();

        String params = this.paramsCode(procInfo);
        String paramsGui = "";
        String msControlCall = "";
        String comma = ", ";
        
        Hashtable ht;
        int c = 0;
        
        String processNameGui = this.getNameGui(processName);
        
        List listProcessesCode = new ArrayList();
        
        /**
         * Builds the parameters for the GUI, that is, the code for the 
         * passage of channels as parameters.
         */
        Iterator it = chanInfoEnv.iteratorKeys();
        
        while (it.hasNext()) {
                
            String channelName = (String) it.next();
            ProcChanUseEnv chanMSEnv = chanInfoEnv.get(channelName);
            
            if (!chanMSEnv.isSync() && !chanUseEnvHid.containsKey(channelName)) {

                /**
                 * Independent channels - they will synchronize only with the gui.
                 * They cannot be hidden otherwise the gui cannot see them.
                 */
                paramsGui = paramsGui + comma + "new GeneralChannel(" + channelName + 
                        ", new Integer(-1))"; // global id of gui is -1
                c++;
            }
        }        
        
        // c is the number of channels that take part in the gui
        paramsGui = "\"" + processName + "\"" + paramsGui;
        
        if (chanInfoEnv.noMultiSync() == 0) {

            /**
             * If there is no multi-synchronization, the main call is the 
             * parallelism of the main process and GUI.
             */
            listProcessesCode.add(this.cspProcessCallCode(processName, params));
            listProcessesCode.add(this.cspProcessCallCode(processNameGui, paramsGui));
            
        } else {
            
            /**
             * If there is multi-synchronization, the main call is the 
             * parallelism of ProcessManagerMultiSync and ControllersManager.
             */
            listProcessesCode.add(this.processManagerCallCode(
                    processName, params, processNameGui, paramsGui));
            listProcessesCode.add(this.controllersManagerCallCode(chanInfoEnv));
            
        }

        /**
         * Creates the parallelism
         */
        code = this.parallelCallRunCode(listProcessesCode, true);
                
        return code;
    }
    
    /**
     * Returns the source code for the call for the constructor of a parallelism.
     *
     * Receives the code for each CSPProcess that takes part in the parallelism.
     *
     * @param   boolean     run     If it is true, appends the call method run() at the end.
     */
    private String parallelCallRunCode(List processes, boolean run) {
        
        String code;
        
        String listProcessesCode = "";
        String comma = "";
        
        Hashtable ht = new Hashtable();
        
        for (int i=0; i < processes.size(); i++) {
        
            listProcessesCode = listProcessesCode + comma + processes.get(i);
            comma = ", ";
        }
        
        ht.put("listProcessesCode", listProcessesCode);
                
        code = Util.getCodeFromTemplate(Constants.TMP_PAR_CALL, ht);
        
        if (run)
            code = code + ".run();";
                
        return code;
    }
   
    /**
     * Returns the code for the call to the constructor of class ProcessManagerMultiSync.java.
     *
     * Typically, this constructor will receive a process which is the parallelism of the main
     * class and the gui.
     *
     */
    private String processManagerCallCode(
            String processName, String params, String processNameGui, String paramsGui) {
     
        String code;
        Hashtable ht = new Hashtable();
        
        List listProcessesCode = new ArrayList();
        listProcessesCode.add(this.cspProcessCallCode(processName, params));
        listProcessesCode.add(this.cspProcessCallCode(processNameGui, paramsGui));
           
        String parallelismCode = this.parallelCallRunCode(listProcessesCode, false);
        
        ht.put("proc", parallelismCode);
        code = Util.getCodeFromTemplate(Constants.TMP_PROC_MNGR_CALL, ht);
        
        return code;
    }
            
        
    /**
     * Returns the code for the call to the constructor of class ControllersManager.java.
     *
     */
    private String controllersManagerCallCode(ChanInfoEnv chanInfoEnv) {
        
        String code;
        String call;
        
        chanInfoEnv.print(0);
        
        Hashtable ht = new Hashtable();
        
        if (chanInfoEnv.noMultiSync() == 1) {

            /**
             * If there is only one controller, this is the call to the 
             * method run of the controller.
             */ 
            Iterator it = chanInfoEnv.iteratorKeys();

            String channelName = "";
            while(it.hasNext()) {
                channelName = (String) it.next();
                ProcChanUseEnv syncEnv = chanInfoEnv.get(channelName);
                if (syncEnv.isMultiSync())
                    break;
            }

            call = "new MultiSyncControl(from_" + channelName + 
                    ", to_" + channelName + ")";
            
        } else {

            /**
             * If there is more than one, this is the call to the method 
             * run of the parallelism of the controllers.
             *
             */ 
            Iterator it = chanInfoEnv.iteratorKeys();
            Hashtable ht2 = new Hashtable();

            List listProcessCode = new ArrayList();
            
            /**
             * iterates over the environments and when it finds one 
             * channel that is multisynchronized, 
             * adds its name to the array
             */
            while(it.hasNext()) {

                String channelName = (String) it.next();
                ProcChanUseEnv syncEnv = chanInfoEnv.get(channelName);
                                
                if (syncEnv.isMultiSync()) {

                    String processCode = "";
                    /**
                     * Multi synchronization. 
                     */
                    ht2 = new Hashtable();
                    ht2.put("channelNm", channelName);

                    processCode = "new MultiSyncControl (from_" + 
                            channelName + ", to_" + channelName + ")";
                    //processCode = Util.getCodeFromTemplate(Constants.TMP_CTRLCALL , ht2);
                    
                    listProcessCode.add(processCode);                   
                }
            }
            
            call = this.parallelCallRunCode(listProcessCode, false);   
        }
        
        ht.put("msControlCall", call);
        code = Util.getCodeFromTemplate(Constants.TMP_CTRL_MNGR_CALL, ht);
            
        return code;
    }
     
    /**
     *
     */
    private String cspProcessCallCode(String name, String paramsCode) {
        return "new " + name + "(" + paramsCode + ")";
    }    
            
    /**
     * Code for the params to the main process in class Main.
     *
     */
    private String paramsCode(ProcInfo procInfo) throws Exception {

        String code = "";
        
        String processName = procInfo.getProcessName();
        CircusProcess circusProcess = procInfo.getProcessPara().getCircusProcess();
        
        List decls;
        String procParamCode = "", chanParamCode;
        
        // parameters of the process
        if (circusProcess instanceof ParamProcess) {
            
            decls = Util.getListTermAnn(circusProcess);
            procParamCode = this.procParamsCode(procInfo.getCodeForActuals());
        }
        
        // visible channels passed as parameters
        chanParamCode = paramsChanCode(procInfo.getProcessPara());
        
        // Channels come first
        if (!chanParamCode.equals("") && !procParamCode.equals(""))
            code = chanParamCode + ", " + procParamCode;
        else
            code = chanParamCode + procParamCode;
        
        return code;
    }    
    
    // Return code for the expressions which are parameters of the process
    private String procParamsCode(List<String> codeForActuals) {
    
        String code = "";
        String comma = "";
        
        for (int i = 0; i < codeForActuals.size(); i++) {
            code = code + comma + codeForActuals.get(i);
            comma = ", ";
        }
        return code;
    }
    
   
    private String paramsChanCode(ProcessPara processParaMain) throws Exception {
        
        String code = "";
        
        String processName = processParaMain.getDeclName().toString();
        CircusProcess circusProcess = processParaMain.getCircusProcess();
        ChanInfoEnv channelMSEnv = Util.getChannelMSEnvAnn(processParaMain);
        Integer id = Util.getIdCircusProcessAnn(circusProcess).getId();
        
        ProcChanEnv procChanEnv = Util.getProcChanEnvAnn(processParaMain);
        ChanUseEnv chanUseEnvHid = procChanEnv.getChanUseEnvHid();
        
        Iterator it = channelMSEnv.iteratorKeys();
        String comma = "";
        
        // For each channel
        while(it.hasNext()) {
            
            String channelName = (String) it.next();
            
            if (!chanUseEnvHid.containsKey(channelName)) {
                
                // Hidden channels are not seen in main
                ProcChanUseEnv syncEnv = channelMSEnv.get(channelName);
                code = code + comma + channelName;
                if (comma.equals("")) comma = ", ";
            }
/**
 * Don't think this is necessary anymore. Why create a new channel
 * for the process with the same id ?
 *            
            if (!syncEnv.isSync()) {
                // This channel will synchronize with the GUI
                ChanUse chanUse = syncEnv.getChanUseGuiNotSyncChannel();
               
                code = code + comma + "new GeneralChannel(" + channelName + 
                        ", new Integer(" + id.intValue() + "))";
            } else {
                
                code = code + comma + channelName;
            }
            
            if (comma.equals("")) comma = ", ";
 */
        }
       
        return code;
    }  
    
    private String chanActPerfWrite(String chanName, CircusType circusType) {
      
        String code = "";

        code += "\n\tprivate void btn_" + chanName + "_ActionPerformed(java.awt.event.ActionEvent evt) {";

        if (circusType.isSyncType()) {
        
            code += "\n\t\t" + chanName + ".write(null);";
            
        } else {

            String chanCommType = circusType.getJavaCircusTypeName();
            
            code += "\n\t\tString st = this.txt_" + chanName + ".getText();";
            code += "\n\t\t" + chanCommType + " o = new " + chanCommType + " (st);";
            code += "\n\t\t" + chanName + ".write(o);";
        }

        code += "\n\t}";
        
        return code;
    }
    
    private String chanActPerfRead(String chanName, CircusType circusType) {
      
        String code = "";

        code += "\n\tprivate void btn_" + chanName + "_ActionPerformed(java.awt.event.ActionEvent evt) {";
        
        if (circusType.isSyncType()) {

            code += "\n\t\t" + chanName + ".read();";
            
        } else {
        
            String chanCommType = circusType.getJavaCircusTypeName();
            code += "\n\t\t" + chanCommType + " o = (" + chanCommType + ") " + chanName + ".read();";
            code += "\n\t\tString st = o.toString();";
            code += "\n\t\tthis.txt_" + chanName + ".setText(st);";
        }
        
        code += "\n\t}";
        
        return code;
    }
    
    /**
     * Code for initialisation of the GUI components for a synchronisation channel.
     */
    private String initGuiChanSync(String chanName) {
      
        String code = "";

        code += "\n\t\tbtn_" + chanName + " = new javax.swing.JButton(\"" + chanName + "\");";
        code += "\n\t\tbtn_" + chanName + ".addActionListener(new java.awt.event.ActionListener() {";        
        code += "\n\t\t\tpublic void actionPerformed(java.awt.event.ActionEvent evt) {";        
        code += "\n\t\t\t\tbtn_" + chanName + "_ActionPerformed(evt);";        
        code += "\n\t\t\t}";        
        code += "\n\t\t});";        
        code += "\n\t\tthis.getContentPane().add(btn_" + chanName + ");";        
        code += "\n\t\tthis.getContentPane().add(new javax.swing.JLabel(\"\"));";
        code += "\n\t\tthis.getContentPane().add(new javax.swing.JLabel(\"\"));";
        code += "\n";        
        
        return code;
    }
    
    /**
     * Code for initialisation of the GUI components for a communication channel.
     */
    private String initGuiChanComm(String chanName, String chanType, 
            String commType) {
        
        String code = "";
        
        code += "\n\t\tbtn_" + chanName + 
                " = new javax.swing.JButton(\"" + chanName + "\");";
        code += "\n\t\ttxt_" + chanName + 
                " = new javax.swing.JTextField();";
        code += "\n\t\tlbl_" + chanName + 
                " = new javax.swing.JLabel(\"" + chanType + ": " + commType + 
                "\", SwingConstants.LEFT);";
        code += "\n\t\tthis.btn_" + chanName + ".addActionListener(new java.awt.event.ActionListener() {";        
        code += "\n\t\t\tpublic void actionPerformed(java.awt.event.ActionEvent evt) {";        
        code += "\n\t\t\t\tbtn_" + chanName + "_ActionPerformed(evt);";        
        code += "\n\t\t\t}";        
        code += "\n\t\t});";        
        code += "\n";        
        code += "\n\t\tthis.getContentPane().add(btn_" + chanName + ");";        
        code += "\n\t\tthis.getContentPane().add(txt_" + chanName + ");";        
        code += "\n\t\tthis.getContentPane().add(lbl_" + chanName + ");";        
        code += "\n";    
        
        return code;
    }
    
    /**
     * Code for imports of a class.
     */
    public String getImports(String projectName) {
        
        String code = "";
        
        code = "import " + projectName + ".axiomaticDefinitions.*;\n";
        code += "import " + projectName + ".processes.*;\n";
        code += "import " + projectName + ".typing.*;\n";
        code += "import jcircusutil.RandomGenerator;\n";
        code += "import jcircusutil.multisync.*;\n";
        code += "import " + projectName + ".gui.*;\n";
        
        return code;
    }
    
    /**
     * Code for package declaration.
     */
    public String getPackage(String projectName, String pack) {
        
        String code = "";
        
        code = "package " + projectName;
        
        if (pack != null)
            code += "." + pack;
        
        code += ";\n\n";
        
        return code;
    }

    /**
     * Returns the name of the GUI class.
     */
    private String getNameGui(String processName) {
        return "Gui_" + processName;
    }

    /**
     * Returns the name of the .bat file.
     */
    private String getNameBat(String processName) {
        return "Run_" + processName;
    }
    
    /**
     * Returns the name of the Main class.
     */
    private String getNameMain(String processName) {
        return "Main_" + processName;
    }

    /**
     * Receives the expression that was entered in the text field and the type
     * that it should correspond. This method is a gambi... It is okay,
     * because of the restrictions on the format of a declaration (the expression
     * must be always a RefExpr). This method performs >>fake<< parsing, typechecking
     * and translation to Java. Come back here in the future, when we will be dealing
     * with more kinds of types and expressions, and change the respective bits
     * of code for calls to the real parser, type checker and translator.
     *
     */
    public String validateAndReturnCodeForParameter(String expression, Expr typeExpr) 
            throws FailParsingException, FailTypeCheckingException {
    
        String code = "";
        String javaTypeName = "";
        if (typeExpr instanceof RefExpr) {
            // For the moment, typeExpr will be always a reference. This will be 
            // changed in the future. Probably this method will receive the actual
            // Type, instead of the type expression.
            String typeExprSt = ((RefExpr) typeExpr).getRefName().toString();
            if (typeExprSt.equals(MathToolkitConstants.NAT) ||
                    typeExprSt.equals(MathToolkitConstants.NUM) ||
                    typeExprSt.equals(MathToolkitConstants.ARITHMOS)) {
                
                // Number
                try {
                    int number = Integer.parseInt(expression);
                    code = "new CircusInteger(" + number + ")";
                    
                } catch (NumberFormatException nfe) {
                    
                    // Simulate parsing here
                    
                    List<String> errors = new Factory().list("Expression is not a number.");
                    throw new FailTypeCheckingException(errors);
                }
                
            } else {
            
                // Free type
                if (environment_.isElementFreeType(typeExprSt, expression)) {
                    
                    // Simulate parsing here
                    
                    code = "new " + typeExprSt + "(" + typeExprSt + "." + expression + ")";
                    
                } else {
                    List<String> errors = new Factory().list("Expression is not an element of free type.");
                    throw new FailTypeCheckingException(errors);
                }
            }

        } else
            throw new InvalidParameterException("");
        
        return code;
    }
                
}

